/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Qt includes
#include <QDebug>


/////////////////////// IsoSpec
#include <IsoSpec++/isoSpec++.h>
#include <IsoSpec++/element_tables.h>


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/libXpertMassCore/IsotopicDataBaseHandler.hpp"

namespace MsXpS
{

namespace libXpertMassCore
{

/*!
\class MsXpS::libXpertMassCore::IsotopicDataBaseHandler
\inmodule libXpertMassCore
\ingroup PolChemDefBuildingdBlocks
\inheaderfile IsotopicDataBaseHandler.hpp

\brief The IsotopicDataBaseHandler class features basic handling of
\l{IsotopicData}.

  The IsotopicDataBaseHandler class provides the skeleton for derived classes
to handle \l{IsotopicData}.

  There are different isotopic data handlers:

  \list
  \li The \l{IsotopicDataLibraryHandler} that handles isotopic data from the
IsoSpec element data tables directly from the library's data. These are the
reference, pristine \e{unmodified} isotopic data.
  \li The \l{IsotopicDataUserConfigHandler} that handles isotopic data with the
exact same format of those from the IsoSpec element data tables. However, these
data correspond to user-modified isotopic data, \e{not} the reference, pristine
\e{unmodified} isotopic data.
  \li The \l{IsotopicDataManualConfigHandler} that handles user-defined data
according to an entirely different format. These data are typically used when
the user defines new chemical elements that cannot fit in the IsoSpec element
data tables format.
  \endlist

  \sa IsotopicDataLibraryHandler, IsotopicDataUserConfigHandler,
IsotopicDataManualConfigHandler
*/

/*!
   \variable int
   MsXpS::libXpertMassCore::IsotopicDataBaseHandler::msp_isotopicData

   \brief The msp_isotopicData is a pointer to \l{IsotopicData}.
*/

/*!
   \variable int MsXpS::libXpertMassCore::IsotopicDataBaseHandler::m_fileName

   \brief The m_fileName is the filename in which to store the \l{IsotopicData}
for from with to load the \l{IsotopicData}.
*/

/*!
  \brief Constructs the \l{IsotopicDataBaseHandler} with \a file_name.

  The msp_isotopicData is initialized by creating an empty IsotopicData
instance.
*/
IsotopicDataBaseHandler::IsotopicDataBaseHandler(const QString &file_name)
  : m_fileName(file_name)
{
  // We cannot allow to have a nullptr msp_isotopicData member.
  msp_isotopicData = std::make_shared<IsotopicData>();
}

/*!
  \brief Constructs the \l{IsotopicDataBaseHandler}.

  \a isotopic_data_sp Isotopic data

  \a file_name File name
*/
IsotopicDataBaseHandler::IsotopicDataBaseHandler(
  IsotopicDataSPtr isotopic_data_sp, const QString &file_name)
  : msp_isotopicData(isotopic_data_sp), m_fileName(file_name)
{
  // We cannot allow to have a nullptr msp_isotopicData member.
  if(msp_isotopicData == nullptr)
    qFatalStream() << "Programming error. The pointer cannot be nullptr.";
}

/*!
  \brief Destructs the \l{IsotopicDataBaseHandler}.
*/
IsotopicDataBaseHandler::~IsotopicDataBaseHandler()
{
  // qDebug();
}

/*!
  \brief Returns the IsotopicData
*/
IsotopicDataSPtr
IsotopicDataBaseHandler::getIsotopicData()
{
  return msp_isotopicData;
}

/*!
  \brief Sets the file name to \a file_name
*/
void
IsotopicDataBaseHandler::setFileName(const QString &file_name)
{
  m_fileName = file_name;
}

/*!
  \brief Returns the file name
*/
QString
IsotopicDataBaseHandler::getFileName()
{
  return m_fileName;
}


} // namespace libXpertMassCore
} // namespace MsXpS
