/* +------------------------------------------------------------------------+
   |                     Mobile Robot Programming Toolkit (MRPT)            |
   |                          https://www.mrpt.org/                         |
   |                                                                        |
   | Copyright (c) 2005-2026, Individual contributors, see AUTHORS file     |
   | See: https://www.mrpt.org/Authors - All rights reserved.               |
   | Released under BSD License. See: https://www.mrpt.org/License          |
   +------------------------------------------------------------------------+ */
#pragma once

#include <mrpt/maps/CPointsMap.h>
#include <mrpt/math/CMatrixF.h>
#include <mrpt/obs/obs_frwds.h>
#include <mrpt/serialization/CSerializable.h>

namespace mrpt::maps
{
/** A cloud of points in 2D or 3D, which can be built from a sequence of laser
 * scans.
 *    This class only stores the coordinates (x,y,z) of each point.
 *
 *  See mrpt::maps::CPointsMap and derived classes for other point cloud
 * classes.
 *
 * \sa CMetricMap, CPoint, mrpt::serialization::CSerializable
 * \ingroup mrpt_maps_grp
 */
class CSimplePointsMap : public CPointsMap
{
  DEFINE_SERIALIZABLE(CSimplePointsMap, mrpt::maps)

 public:
  /** Default constructor */
  CSimplePointsMap() = default;
  explicit CSimplePointsMap(const CPointsMap& o) { CPointsMap::operator=(o); }
  CSimplePointsMap(const CSimplePointsMap& o) : CPointsMap() { CPointsMap::operator=(o); }
  CSimplePointsMap& operator=(const CPointsMap& o)
  {
    CPointsMap::operator=(o);
    return *this;
  }
  CSimplePointsMap& operator=(const CSimplePointsMap& o)
  {
    impl_copyFrom(o);
    return *this;
  }

  // --------------------------------------------
  /** @name Pure virtual interfaces to be implemented by any class derived
   from CPointsMap
    @{ */
  void reserve(size_t newLength) override;  // See base class docs
  void resize(size_t newLength) override;   // See base class docs
  void setSize(size_t newLength) override;  // See base class docs

  /** Get all the data fields for one point as a vector: [X Y Z]
   *  Unlike getPointAllFields(), this method does not check for index out of
   * bounds
   * \sa getPointAllFields, setPointAllFields, setPointAllFieldsFast
   */
  void getPointAllFieldsFast(size_t index, std::vector<float>& point_data) const override
  {
    point_data.resize(3);
    point_data[0] = m_x[index];
    point_data[1] = m_y[index];
    point_data[2] = m_z[index];
  }
  /** Set all the data fields for one point as a vector: [X Y Z]
   *  Unlike setPointAllFields(), this method does not check for index out of
   * bounds
   * \sa setPointAllFields, getPointAllFields, getPointAllFieldsFast
   */
  void setPointAllFieldsFast(size_t index, const std::vector<float>& point_data) override
  {
    ASSERTDEB_(point_data.size() == 3);
    m_x[index] = point_data[0];
    m_y[index] = point_data[1];
    m_z[index] = point_data[2];
  }

  // See CPointsMap::loadFromRangeScan()
  void loadFromRangeScan(
      const mrpt::obs::CObservation2DRangeScan& rangeScan,
      const std::optional<const mrpt::poses::CPose3D>& robotPose) override;
  // See CPointsMap::loadFromRangeScan()
  void loadFromRangeScan(
      const mrpt::obs::CObservation3DRangeScan& rangeScan,
      const std::optional<const mrpt::poses::CPose3D>& robotPose) override;

 protected:
  // Friend methods:
  template <class Derived>
  friend struct detail::loadFromRangeImpl;
  template <class Derived>
  friend struct detail::pointmap_traits;

 public:
  /** @} */

  // See base docs
  const mrpt::maps::CSimplePointsMap* getAsSimplePointsMap() const override { return this; }

 protected:
  /** Clear the map, erasing all the points.
   */
  void internal_clear() override;

  /** @name PLY Import virtual methods to implement in base classes
    @{ */
  /** In a base class, reserve memory to prepare subsequent calls to
   * PLY_import_set_vertex */
  void PLY_import_set_vertex_count(size_t N) override;
  void PLY_import_set_vertex_timestamp(
      [[maybe_unused]] size_t idx, [[maybe_unused]] const double unixTimestamp) override
  {
    // do nothing, this class ignores timestamps
  }
  /** @} */

  MAP_DEFINITION_START(CSimplePointsMap)
  /** Observations insertion options */
  mrpt::maps::CPointsMap::TInsertionOptions insertionOpts;
  /** Probabilistic observation likelihood options */
  mrpt::maps::CPointsMap::TLikelihoodOptions likelihoodOpts;
  /** Rendering as 3D object options */
  mrpt::maps::CPointsMap::TRenderOptions renderOpts;
  MAP_DEFINITION_END(CSimplePointsMap)
};  // End of class def.
}  // namespace mrpt::maps

namespace mrpt::opengl
{
template <>
class PointCloudAdapter<mrpt::maps::CSimplePointsMap> :
    public PointCloudAdapter<mrpt::maps::CPointsMap>
{
 public:
  explicit PointCloudAdapter(const mrpt::maps::CSimplePointsMap& pts) :
      PointCloudAdapter<mrpt::maps::CPointsMap>(pts)
  {
  }
};
}  // namespace mrpt::opengl
