/*
 * Copyright (c) 2003, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jvmti.PopFrame;

import nsk.share.Wicket;
import java.io.PrintStream;

/**
 * This test checks that a frame can be popped and<br>
 * no JVMTI events would be generated by the JVMTI function
 * <code>PopFrame()</code>.<br>
 * The test creates an instance of inner class <code>popFrameCls</code>
 * and start it in a separate thread. Then the test pops frame
 * from the thread's stack of the class <code>popFrameCls</code>.
 */
public class popframe001 {
    static final int PASSED = 0;
    static final int FAILED = 2;
    static final int JCK_STATUS_BASE = 95;

    static volatile boolean popFdone = false;
    static volatile int totRes = PASSED;
    private PrintStream out;
    private popFrameCls popFrameClsThr;
    static Object barrier = new Object();
    static Wicket startingBarrier;

    static {
        try {
            System.loadLibrary("popframe001");
        } catch (UnsatisfiedLinkError e) {
            System.err.println("Could not load popframe001 library");
            System.err.println("java.library.path:" +
                System.getProperty("java.library.path"));
            throw e;
        }
    }

    native static int doPopFrame(popFrameCls popFrameClsThr);
    native static int suspThread(popFrameCls popFrameClsThr);
    native static int resThread(popFrameCls popFrameClsThr);

    public static void main(String[] argv) {
        argv = nsk.share.jvmti.JVMTITest.commonInit(argv);

        System.exit(run(argv, System.out) + JCK_STATUS_BASE);
    }

    public static int run(String argv[], PrintStream out) {
        return new popframe001().runIt(argv, out);
    }

    private int runIt(String argv[], PrintStream out) {
        int retValue = 0;

        this.out = out;

        popFrameClsThr = new popFrameCls("Tested Thread");
        startingBarrier = new Wicket();
        // start the child thread
        popFrameClsThr.start();
        startingBarrier.waitFor();
        // pause until the child thread exits notification-block
        synchronized (barrier) {
        }

        out.println("Going to suspend the thread...");
        retValue = suspThread(popFrameClsThr);
        if (retValue != PASSED) {
            out.println("TEST: failed to suspend thread");
            return FAILED;
        }

        // pop a frame of the child thread
        out.println("Going to pop a frame...");
        retValue = doPopFrame(popFrameClsThr);
        popFdone = true;
        popFrameClsThr.letItGo();
        if (retValue != PASSED) {
            out.println("TEST: failed to pop frame");
            resThread(popFrameClsThr);
            return FAILED;
        }

        out.println("Going to resume the thread...");
        retValue = resThread(popFrameClsThr);
        if (retValue != PASSED) {
            out.println("TEST: failed to resume thread");
            return FAILED;
        }

        try {
            popFrameClsThr.join();
        } catch (InterruptedException e) {
            out.println("TEST INCOMPLETE: caught " + e);
            return FAILED;
        }

        return totRes;
    }

    class popFrameCls extends Thread {
        private volatile boolean flag = true;

        popFrameCls(String name) {
            super(name);
        }

        public void run() {
            activeMethod();
            out.println("popFrameCls (" + this + "): exiting...");
        }

        public void activeMethod() {
            boolean compl = true; // complain in a finally block

            if (popframe001.popFdone) { // popping has been done
                out.println("popFrameCls (" + this + "): enter activeMethod() after popping");
                return;
            }
            try {
                // notify the main thread
                synchronized (popframe001.barrier) {
                    out.println("popFrameCls (" + this + "): notifying main thread");
                    popframe001.startingBarrier.unlock();
                    out.println("popFrameCls (" + this + "): inside activeMethod()");
                }
                // loop until the main thread pops us
                int i = 0;
                int n = 1000;
                while (flag) {
                    if (n <= 0) {
                        n = 1000;
                    }
                    if (i > n) {
                        i = 0;
                        n--;
                    }
                    i++;
                }
                if (popframe001.popFdone) { // popping has been done
                    out.println("TEST FAILED: a tested frame has not been really popped");
                    popframe001.totRes = FAILED;
                    compl = false;
                }
            } catch (Exception e) {
                out.println("FAILURE: popFrameCls (" + this + "): caught " + e);
                popframe001.totRes = FAILED;
                compl = false;
            } finally {
                if (compl) {
                    out.println("TEST FAILED: finally block was executed after PopFrame()");
                    popframe001.totRes = FAILED;
                }
            }
        }

        public void letItGo() {
            flag = false;
        }
    }
}
